#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.DAL.UnitTests
{
	[TestFixture]
	public class DAL_OrderedTest : BaseTest 
	{
		
		private DataTable _dtOrderableTest;
		private DataTable _dtOrderedTest;
		private DataTable _dtSpecimenTest;
		private DataTable _dtPatientSpecimen;
		private DataTable _dtCannedComment;
		private DataTable _dtPatientTreatment;
		private DataTable _dtBloodUnit;

		[SetUp]
		protected void SetUp() 
		{				
			this.BaseSetUp();

			if (this.RefreshData)
			{
				this._dtOrderableTest = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM OrderableTest", true);

				//Get SpecimenTest data ***********************************************************
				this._dtSpecimenTest = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM SpecimenTest WHERE RecordStatusCode = 'A' AND OrderedTestGuid IN (SELECT OrderedTestGuid FROM OrderedTest WHERE PatientSpecimenGuid IS NOT NULL AND OrderableTestId IN (1,3,7,8,9,10,113,118,119) AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "')", false);
				if (this._dtSpecimenTest.Rows.Count == 0)
				{
					new BOL.UnitTests.BOL_SpecimenTest().CreateSpecimenTestResults();
					this._dtSpecimenTest = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM SpecimenTest WHERE RecordStatusCode = 'A' AND OrderedTestGuid IN (SELECT OrderedTestGuid FROM OrderedTest WHERE PatientSpecimenGuid IS NOT NULL AND OrderableTestId IN (1,3,7,8,9,10,113,118,119) AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "')", true);
				}

				//Get OrderedTest data ***********************************************************
				this._dtOrderedTest = UnitTestUtil.RunSQLGetDataTable("SELECT * FROM OrderedTest WHERE OrderedTestGuid = '" + this._dtSpecimenTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid].ToString() + "'", true);

				//Get PatientSpecimen data ***********************************************************
				this._dtPatientSpecimen = UnitTestUtil.RunSQLGetDataTable("SELECT * FROM PatientSpecimen WHERE PatientSpecimenGuid = '" + this._dtOrderedTest.Rows[0][TABLES.OrderedTest.PatientSpecimenGuid].ToString() + "'", true);

				//Get PatientTreatment data ***********************************************************
				this._dtPatientTreatment = UnitTestUtil.RunSQLGetDataTable("SELECT * FROM PatientTreatment WHERE PatientGuid = '" + this._dtPatientSpecimen.Rows[0][TABLES.PatientSpecimen.PatientGuid].ToString() + "'", true);

				//Get CannedComment data ***********************************************************
				this._dtCannedComment = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM CannedComment WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);

				//Get BloodUnit data ***********************************************************
				this._dtBloodUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM BloodUnit WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", false);
				if (this._dtBloodUnit.Rows.Count == 0)
				{
					//Being lazy for now, the above specimentest creation should satisfy everything
					BOL.BloodUnit newUnit = DataCreator.CreateRBCBloodUnit(true);
					DataCreator.CreateAboRhConfirmationTestsForUnit(newUnit.BloodUnitGuid, newUnit.BloodUnitMedia.AboRh);
					DataCreator.CreatePatientAssignment(newUnit.BloodUnitGuid, Guid.NewGuid(), true, true);
					this._dtBloodUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM BloodUnit WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'", true);
				}
				this.RefreshData = false;
			}
		}


		[Test]
		public void IsOrderedTestPending_Pass()
		{	
			bool retValue = DAL.OrderedTest.IsOrderedTestPending();

			//We may or may not have tests pending, so a return value is a pass
			Assert.IsTrue(true, "ReturnValue");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IsOrderedTestPending_Fail()
		{			
		}

		[Test]
		public void GetOrderableTestList_Pass()
		{
			DataTable dtTest = DAL.OrderedTest.GetOrderableTestList();

			Assert.AreEqual(true, dtTest.Rows.Count > 0, "Rows.Count");
			Assert.AreEqual(3, dtTest.Columns.Count, "Columns.Count");
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestId), TABLES.OrderableTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestName), TABLES.OrderableTest.OrderableTestName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestText), TABLES.OrderableTest.OrderableTestText);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetOrderableTestList_Fail()
		{
		}

		[Test]
		public void GetReorderableTestList_Pass()
		{
			int unitId = System.Convert.ToInt32(this._dtOrderableTest.Rows[0][TABLES.OrderableTest.OrderableTestId]);

			DataTable dtTest = DAL.OrderedTest.GetReorderableTestList(unitId);

			Assert.AreEqual(true, dtTest.Rows.Count > 0, "Rows.Count");
			Assert.AreEqual(3, dtTest.Columns.Count, "Columns.Count");
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestId), TABLES.OrderableTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestName), TABLES.OrderableTest.OrderableTestName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestText), TABLES.OrderableTest.OrderableTestText);
		}

		[Test]
		public void GetReorderableTestList_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetReorderableTestList(0);

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetOrderedTest_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];

			DataTable dtTest = DAL.OrderedTest.GetOrderedTest(unitGuid);

			Assert.AreEqual(true, dtTest.Rows.Count > 0, "Rows.Count");
			Assert.AreEqual(26, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestGuid), TABLES.OrderedTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.PatientOrderGuid), TABLES.OrderedTest.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CprsOrderNumber), TABLES.OrderedTest.CprsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.VbecsOrderNumber), TABLES.OrderedTest.VbecsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RelatedComponentGuid), TABLES.OrderedTest.RelatedComponentGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.SpecimenRequiredIndicator), TABLES.OrderedTest.SpecimenRequiredIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.PatientSpecimenGuid), TABLES.OrderedTest.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderStatusCode), TABLES.OrderedTest.OrderStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderUrgencyCode), TABLES.OrderedTest.OrderUrgencyCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestComments), TABLES.OrderedTest.OrderedTestComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RequiredResultsDate), TABLES.OrderedTest.RequiredResultsDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderableTestId), TABLES.OrderedTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TestInterpretation), TABLES.OrderedTest.TestInterpretation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TestInterpretationComments), TABLES.OrderedTest.TestInterpretationComments);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.AutoInstrumentName), TABLES.OrderedTest.AutoInstrumentName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TaskStatusCode), TABLES.OrderedTest.TaskStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RelatedTestGuid), TABLES.OrderedTest.RelatedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CancellationCannedCommentGuid), TABLES.OrderedTest.CancellationCannedCommentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CancellationText), TABLES.OrderedTest.CancellationText);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabOrderNumber), TABLES.OrderedTest.LabOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabTestId), TABLES.OrderedTest.LabTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.DivisionCode), TABLES.OrderedTest.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LastUpdateDate), TABLES.OrderedTest.LastUpdateDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LastUpdateUser), TABLES.OrderedTest.LastUpdateUser);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientTreatmentGuid), TABLES.PatientTreatment.PatientTreatmentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RowVersion), TABLES.OrderedTest.RowVersion);
		}

		[Test]
		public void GetOrderedTest_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetOrderedTest(System.Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void SaveReflexTests_Pass()
		{
			this.RefreshData = true;

			DataSet dsSave = new DataSet();
		
			//*******BUILD THE DATA
			System.Guid poGuid = System.Guid.NewGuid();
			System.Guid otGuid = System.Guid.NewGuid();
			System.Guid ouGuid = System.Guid.NewGuid();
			System.Guid buGuid = (System.Guid)this._dtBloodUnit.Rows[0][TABLES.OrderedUnit.BloodUnitGuid];
			System.Guid ptGuid = (System.Guid)this._dtPatientTreatment.Rows[0][TABLES.PatientOrder.PatientTreatmentGuid];

			DataTable dtInsPO = SprocHelper.GetInsertPatientOrderSprocTable(poGuid, ptGuid, false);
			DataTable dtInsOT = SprocHelper.GetInsertOrderedTestSprocTable(otGuid, poGuid, System.Guid.Empty);
			DataTable dtInsOU = SprocHelper.GetInsertOrderedUnitSprocTable(ouGuid, System.Guid.Empty, buGuid);

			//Blank out the Unwanted fields
			dtInsOT.Rows[0][TABLES.PatientSpecimen.PatientSpecimenGuid] = DBNull.Value;
			dtInsOU.Rows[0][TABLES.OrderedUnit.OrderedComponentGuid] = DBNull.Value;

			dsSave.Tables.Add(dtInsPO);
			dsSave.Tables.Add(dtInsOT);
			dsSave.Tables.Add(dtInsOU);

			DataTable dtException = DAL.ExceptionReport.GetExceptionReportReflexTest(this.TestDivisionCode, (System.Guid)this._dtPatientTreatment.Rows[0][TABLES.Patient.PatientGuid]).Tables[0].Copy();
			dsSave.Tables.Add(dtException);

			bool retValue = DAL.OrderedTest.SaveReflexTests(dsSave);

			Assert.AreEqual(true, retValue);
		}

		[Test]
		[ExpectedException(typeof(Common.DataAccessLayerException))]
		public void SaveReflexTests_Fail()
		{
			this.RefreshData = true;

			DataSet dsSave = new DataSet();
		
			//*******BUILD THE DATA
			System.Guid poGuid = System.Guid.NewGuid();
			System.Guid otGuid = System.Guid.NewGuid();
			System.Guid ouGuid = System.Guid.NewGuid();
			System.Guid buGuid = System.Guid.NewGuid();
			System.Guid ptGuid = System.Guid.NewGuid();

			DataTable dtInsPO = SprocHelper.GetInsertPatientOrderSprocTable(poGuid, ptGuid, true);
			DataTable dtInsOT = SprocHelper.GetInsertOrderedTestSprocTable(otGuid, poGuid, System.Guid.Empty);
			DataTable dtInsOU = SprocHelper.GetInsertOrderedUnitSprocTable(ouGuid, System.Guid.Empty, buGuid);

			dsSave.Tables.Add(dtInsPO);
			dsSave.Tables.Add(dtInsOT);
			dsSave.Tables.Add(dtInsOU);

			DAL.OrderedTest.SaveReflexTests(dsSave);
		}

		
		[Test]
		public void GetTestingTaskList_Pass()
		{
			DataTable dtTest = DAL.OrderedTest.GetTestingTaskList(string.Empty);

			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");

			Assert.AreEqual(38, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderUrgency.OrderUrgencyText), TABLES.OrderUrgency.OrderUrgencyText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestGuid), TABLES.OrderedTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderUrgencyCode), TABLES.OrderedTest.OrderUrgencyCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestComments), TABLES.OrderedTest.OrderedTestComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CprsOrderNumber), TABLES.OrderedTest.CprsOrderNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderableTestId), TABLES.OrderedTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RequiredResultsDate), TABLES.OrderedTest.RequiredResultsDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TaskStatusCode), TABLES.OrderedTest.TaskStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RelatedTestGuid), TABLES.OrderedTest.RelatedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RelatedComponentGuid), TABLES.OrderedTest.RelatedComponentGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.SpecimenAccessionNumber), TABLES.OrderedTest.SpecimenAccessionNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabTestId), TABLES.OrderedTest.LabTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LastUpdateUser), TABLES.OrderedTest.LastUpdateUser);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RowVersion), TABLES.OrderedTest.RowVersion);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientGuid), TABLES.Patient.PatientGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.VistaPatientId), TABLES.Patient.VistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSsn), TABLES.Patient.PatientSsn);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientLocation), TABLES.PatientTreatment.PatientLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientTreatmentGuid), TABLES.PatientTreatment.PatientTreatmentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.BloodTypeCode), TABLES.Patient.BloodTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.RhFactorCode), TABLES.Patient.RhFactorCode);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.PatientOrderGuid), TABLES.PatientOrder.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderFirstName), TABLES.PatientOrder.OrderingProviderFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderLastName), TABLES.PatientOrder.OrderingProviderLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderMiddleInitial), TABLES.PatientOrder.OrderingProviderMiddleInitial);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderPlacedDatetime), TABLES.PatientOrder.OrderPlacedDatetime);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderReceivedDatetime), TABLES.PatientOrder.OrderReceivedDatetime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.PatientSpecimenGuid), TABLES.PatientSpecimen.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.SpecimenUid), TABLES.PatientSpecimen.SpecimenUid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.SpecimenExpirationDate), TABLES.PatientSpecimen.SpecimenExpirationDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestName), TABLES.OrderableTest.OrderableTestName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TaskStatus.TaskStatusText), TABLES.TaskStatus.TaskStatusText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderUrgency.SortColumn), TABLES.OrderUrgency.SortColumn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderGroupNumber), TABLES.PatientOrder.OrderGroupNumber);
		}

		[Test]
		public void GetTestingTaskList_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetTestingTaskList("Thoriin");

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}
		
		[Test]
		public void GetTaskStatusList_Pass()
		{
			DataTable dtTest = DAL.OrderedTest.GetTaskStatusList();
			Assert.AreEqual(true, dtTest.Rows.Count > 0, "Rows.Count");
			Assert.AreEqual(5, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TaskStatus.TaskStatusCode), TABLES.TaskStatus.TaskStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TaskStatus.TaskStatusText), TABLES.TaskStatus.TaskStatusText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TaskStatus.SortColumn), TABLES.TaskStatus.SortColumn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TaskStatus.LastUpdateUser), TABLES.TaskStatus.LastUpdateUser);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TaskStatus.RowVersion), TABLES.TaskStatus.RowVersion);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetTaskStatusList_Fail()
		{
		}

		[Test]
		public void GetOrderedTestsforSpecimen_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.PatientSpecimenGuid];

			DataTable dtTest = DAL.OrderedTest.GetOrderedTestsforSpecimen(unitGuid);

			Assert.AreEqual(true, dtTest.Rows.Count > 0, "Rows.Count");
			Assert.AreEqual(8, dtTest.Columns.Count, "Columns.Count");
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.RequestedProcedure), ARTIFICIAL.RequestedProcedure);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RequiredResultsDate), TABLES.OrderedTest.RequiredResultsDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CprsOrderNumber), TABLES.OrderedTest.CprsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TaskStatus.TaskStatusText), TABLES.TaskStatus.TaskStatusText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TaskStatusCode), TABLES.OrderedTest.TaskStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.PatientOrderGuid), TABLES.OrderedTest.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestGuid), TABLES.OrderedTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RowVersion), TABLES.OrderedTest.RowVersion);
		}

		[Test]
		public void GetOrderedTestsforSpecimen_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetOrderedTestsforSpecimen(System.Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetTestOrders_Pass()
		{
			DataTable dtTest = DAL.OrderedTest.GetTestOrders();

			Assert.AreEqual(true, dtTest.Rows.Count >= 0, "Rows.Count");
			Assert.AreEqual(48, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.PatientOrderGuid), TABLES.PatientOrder.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientGuid), TABLES.PatientTreatment.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderGroupNumber), TABLES.PatientOrder.OrderGroupNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderLastName), TABLES.PatientOrder.OrderingProviderLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderFirstName), TABLES.PatientOrder.OrderingProviderFirstName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderEnteredByLastName), TABLES.PatientOrder.OrderEnteredByLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderEnteredByFirstName), TABLES.PatientOrder.OrderEnteredByFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderComments), TABLES.PatientOrder.OrderComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingHospitalLocation), TABLES.PatientOrder.OrderingHospitalLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.SurgeryName), TABLES.PatientOrder.SurgeryName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.InformedConsentIndicator), TABLES.PatientOrder.InformedConsentIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.RequestReason), TABLES.PatientOrder.RequestReason);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderReceivedDatetime), TABLES.PatientOrder.OrderReceivedDatetime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSsn), TABLES.Patient.PatientSsn);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.BloodTypeCode), TABLES.Patient.BloodTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.RhFactorCode), TABLES.Patient.RhFactorCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientTreatingFacility), TABLES.PatientTreatment.PatientTreatingFacility);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientRoomBed), TABLES.PatientTreatment.PatientRoomBed);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientAdmittingDiagnosis), TABLES.PatientTreatment.PatientAdmittingDiagnosis);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientLocation), TABLES.PatientTreatment.PatientLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TreatingSpecialty.TreatingSpecialtyName), TABLES.TreatingSpecialty.TreatingSpecialtyName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientServiceBranch), TABLES.Patient.PatientServiceBranch);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.VistaPatientId), TABLES.Patient.VistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.VbecsOrderNumber), TABLES.OrderedTest.VbecsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.PatientSpecimenGuid), TABLES.OrderedTest.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestGuid), TABLES.OrderedTest.OrderedTestGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CprsOrderNumber), TABLES.OrderedTest.CprsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderUrgencyCode), TABLES.OrderedTest.OrderUrgencyCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestComments), TABLES.OrderedTest.OrderedTestComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RequiredResultsDate), TABLES.OrderedTest.RequiredResultsDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderStatusCode), TABLES.OrderedTest.OrderStatusCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderableTestId), TABLES.OrderedTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TestInterpretationComments), TABLES.OrderedTest.TestInterpretationComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.SpecimenRequiredIndicator), TABLES.OrderedTest.SpecimenRequiredIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.DivisionCode), TABLES.OrderedTest.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabOrderNumber), TABLES.OrderedTest.LabOrderNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabTestId), TABLES.OrderedTest.LabTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RowVersion), TABLES.OrderedTest.RowVersion);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.SpecimenUid), TABLES.PatientSpecimen.SpecimenUid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderUrgency.OrderUrgencyText), TABLES.OrderUrgency.OrderUrgencyText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderStatus.OrderStatusText), TABLES.OrderStatus.OrderStatusText);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestName), TABLES.OrderableTest.OrderableTestName);
			Assert.AreEqual(true, dtTest.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.UrgencySortColumn), Common.DatabaseConstants.ArtificialColumnNames.UrgencySortColumn);
			Assert.AreEqual(true, dtTest.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.RequestSortColumn), Common.DatabaseConstants.ArtificialColumnNames.RequestSortColumn);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetTestOrders_Fail()
		{
		}

		[Test]
		public void GetTestResultsForTask_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];

			DataTable dtTest = DAL.OrderedTest.GetTestResultsForTask(unitGuid);
			Assert.AreEqual(17, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.BloodTestTypeId), TABLES.BloodUnitTest.BloodTestTypeId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.BloodTestName), TABLES.BloodTestType.BloodTestName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestResultId), TABLES.SpecimenTest.TestResultId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestComments), TABLES.SpecimenTest.TestComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestDate), TABLES.SpecimenTest.TestDate);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.DivisionCode), TABLES.SpecimenTest.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestTechId), TABLES.SpecimenTest.TestTechId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.EntryTechId), TABLES.SpecimenTest.EntryTechId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.CorrectedResultIndicator), TABLES.SpecimenTest.CorrectedResultIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.SpecimenTestGuid), TABLES.SpecimenTest.SpecimenTestGuid);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.ReactivityPhaseCode), TABLES.SpecimenTest.ReactivityPhaseCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.RowVersion), TABLES.SpecimenTest.RowVersion);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.RackGuid), TABLES.SpecimenTest.RackGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.AboRhChangeIndicator), TABLES.SpecimenTest.AboRhChangeIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderableTestId), TABLES.OrderedTest.OrderableTestId);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestName), TABLES.OrderableTest.OrderableTestName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.SpecimenUid), TABLES.PatientSpecimen.SpecimenUid);
		}


		[Test]
		public void GetTestResultsForTask_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetTestResultsForTask(System.Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetTestResultsWithErrorForTask_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];

			DataTable dtTest = DAL.OrderedTest.GetTestResultsWithErrorForTask(unitGuid);
			Assert.AreEqual(3, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.BloodTestTypeId), TABLES.BloodUnitTest.BloodTestTypeId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.BloodTestName), TABLES.BloodTestType.BloodTestName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestResultId), TABLES.SpecimenTest.TestResultId);
		}

		[Test]
		public void GetTestResultsWithErrorForTask_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetTestResultsWithErrorForTask(System.Guid.NewGuid());
			Assert.IsTrue(dtTest.Rows.Count == 0);
		}


		[Test]
		public void SaveTestResults_Pass()
		{
			this.RefreshData = true;

			DataSet dsSave = new DataSet();

			System.Guid stGuid = System.Guid.NewGuid();
			System.Guid otGuid = DataCreator.GetRecordGuid("SELECT TOP 1 OrderedTestGuid FROM OrderedTest WHERE DivisionCode = '" + this.TestDivisionCode + "'");
			System.Guid psGuid = DataCreator.GetRecordGuid("SELECT TOP 1 PatientSpecimenGuid FROM PatientSpecimen WHERE DivisionCode = '" + this.TestDivisionCode + "'");
			int bloodTestTypeId = (int)Common.TestType.BCells;

			//spInsertSpecimenTest
			DataTable dtIST = SprocHelper.GetInsertSpecimenTestSprocTable(stGuid, otGuid, psGuid, bloodTestTypeId);
			dtIST.TableName = "SpecimenTestInsert";
			dsSave.Tables.Add(dtIST);

			//spUpdateSpecimenTestForPatientTesting
			DataRow drTemp = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM SpecimenTest WHERE DivisionCode = '" + this.TestDivisionCode + "'");
			stGuid = (System.Guid)drTemp[TABLES.SpecimenTest.SpecimenTestGuid];
			byte[] rowVersion = (byte[])drTemp[TABLES.SpecimenTest.RowVersion];
			bloodTestTypeId = System.Convert.ToInt16(drTemp[TABLES.SpecimenTest.BloodTestTypeId]);
			DataTable dtUSTFPT = SprocHelper.GetUpdateSpecimenTestForPatientTestingSprocTable(stGuid, rowVersion, bloodTestTypeId);
			dtUSTFPT.TableName = "SpecimenTestUpdate";
			dsSave.Tables.Add(dtUSTFPT);

			//spInsertBloodUnitTest
			System.Guid butGuid = System.Guid.NewGuid();
			System.Guid buGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid FROM BloodUnit WHERE DivisionCode = '" + this.TestDivisionCode + "'");
			DataTable dtIBUT = SprocHelper.GetInsertBloodUnitTestSprocTable(butGuid, buGuid, bloodTestTypeId, "1");
			dtIBUT.TableName = "BloodUnitTestInsert";
			dsSave.Tables.Add(dtIBUT);

			//spUpdateBloodUnitTestResult
			drTemp = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM BloodUnitTest WHERE DivisionCode = '" + this.TestDivisionCode + "'");
			butGuid = (System.Guid)drTemp[TABLES.BloodUnitTest.BloodUnitTestGuid];
			rowVersion = (byte[])drTemp[TABLES.BloodUnitTest.RowVersion];
			DataTable dtUBUTR = SprocHelper.GetUpdateBloodUnitTestResultSprocTable(butGuid, rowVersion);
			dtUBUTR.TableName = "BloodUnitTestUpdate";
			dsSave.Tables.Add(dtUBUTR);

			//spUpdateOrderedTestTaskStatus
			drTemp = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM OrderedTest WHERE TaskStatusCode IS NOT NULL AND DivisionCode = '" + this.TestDivisionCode + "'");
			otGuid = (System.Guid)drTemp[TABLES.OrderedTest.OrderedTestGuid];
			rowVersion = (byte[])drTemp[TABLES.OrderedTest.RowVersion];
			string taskStatusCode = drTemp[TABLES.OrderedTest.TaskStatusCode].ToString();
			DataTable dtUOTTS = SprocHelper.GetUpdateOrderedTestTaskStatusSprocTable(otGuid, taskStatusCode, rowVersion);
			dsSave.Tables.Add(dtUOTTS);

			//spUpdateRackLot
			drTemp = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM RackLot WHERE RecordStatusCode = 'A' AND DivisionCode = '" + this.TestDivisionCode + "'");
			System.Guid rlGuid = (System.Guid)drTemp[TABLES.RackLot.RackLotGuid];
			System.Guid reagentGuid = (System.Guid)drTemp[TABLES.RackLot.ReagentGuid];
			rowVersion = (byte[])drTemp[TABLES.RackLot.RowVersion];
			DataTable dtURL = SprocHelper.GetUpdateRackLotSprocTable(rlGuid, reagentGuid, rowVersion);
			dsSave.Tables.Add(dtURL);

			//spInsertRackTest
			System.Guid rtGuid = System.Guid.NewGuid();
			DataTable dtIRT = SprocHelper.GetInsertRackTestSprocTable(rtGuid, rlGuid);
			dtIRT.TableName = "RackTestInsert";
			dsSave.Tables.Add(dtIRT);

			//spUpdateRackTest
			drTemp = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM RackTest WHERE RecordStatusCode = 'A' AND DivisionCode = '" + this.TestDivisionCode + "'");
			rtGuid = (System.Guid)drTemp[TABLES.RackTest.RackTestGuid];
			rowVersion = (byte[])drTemp[TABLES.RackTest.RowVersion];
			DataTable dtURT = SprocHelper.GetUpdateRackTestSprocTable(rtGuid, rowVersion);
			dtURT.TableName = "RackTestUpdate";
			dsSave.Tables.Add(dtURT);

			//spInsertAntiseraOrderedTest
			System.Guid aotGuid = System.Guid.NewGuid();
			rlGuid = DataCreator.GetRecordGuid("SELECT TOP 1 RackLotGuid FROM RackLot WHERE RecordStatusCode = 'A' AND DivisionCode = '" + this.TestDivisionCode + "'");
			otGuid = DataCreator.GetRecordGuid("SELECT TOP 1 OrderedTestGuid FROM OrderedTest WHERE DivisionCode = '" + this.TestDivisionCode + "'");
			DataTable dtIAOT = SprocHelper.GetInsertAntiseraOrderedTestSprocTable(aotGuid, otGuid, rlGuid);
			dsSave.Tables.Add(dtIAOT);

			//spInsertTransfusionRequirements
			System.Guid ptrGuid = System.Guid.NewGuid();
			System.Guid patientGuid = DataCreator.GetRecordGuid("SELECT TOP 1 PatientGuid FROM PatientTreatment");
			DataTable dtITR = SprocHelper.GetInsertTransfusionRequirements(ptrGuid, patientGuid);
			dsSave.Tables.Add(dtITR);

			//spUpdateOrderedUnit
			drTemp = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM OrderedUnit WHERE RecordStatusCode = 'A' AND DivisionCode = '" + this.TestDivisionCode + "'");
			System.Guid ouGuid = (System.Guid)drTemp[TABLES.OrderedUnit.OrderedUnitGuid];
			rowVersion = (byte[])drTemp[TABLES.OrderedUnit.RowVersion];
			DataTable dtUOU = SprocHelper.GetUpdateOrderedUnitSprocTable(ouGuid, rowVersion);
			dsSave.Tables.Add(dtUOU);

			//spInsertExceptionQC
			DataTable dtIEQC = DAL.ExceptionReport.GetExceptionReportQC(UnitTestConfig.TestDivisionCode, patientGuid).Tables[0].Copy();
			dsSave.Tables.Add(dtIEQC);

			//spInsertExceptionQCNotPerformed
			DataTable dtIEQCNP = DAL.ExceptionReport.GetExceptionReportQCNotPerformed(UnitTestConfig.TestDivisionCode, patientGuid).Tables[0].Copy();
			dsSave.Tables.Add(dtIEQCNP);

			//spInsertExceptionPatientAboRHDiscrepancy
			DataTable dtIEPARD = DAL.ExceptionReport.GetExceptionReportPatientAboRHDiscrepancy(UnitTestConfig.TestDivisionCode, patientGuid).Tables[0].Copy();
			dsSave.Tables.Add(dtIEPARD);

			//spInsertExceptionPatientAntigenDiscrepancy
			DataTable dtIEPAD = DAL.ExceptionReport.GetExceptionReportPatientAntigenDiscrepancy(UnitTestConfig.TestDivisionCode, patientGuid).Tables[0].Copy();
			dsSave.Tables.Add(dtIEPAD);

			//spInsertExceptionPatientAboRH
			DataTable dtIEPAR = DAL.ExceptionReport.GetExceptionReportPatientAboRH(UnitTestConfig.TestDivisionCode, patientGuid).Tables[0].Copy();
			dsSave.Tables.Add(dtIEPAR);

			//spInsertExceptionPatientTest
			DataTable dtIEPT = DAL.ExceptionReport.GetExceptionReportPatientTest(UnitTestConfig.TestDivisionCode, patientGuid).Tables[0].Copy();
			dsSave.Tables.Add(dtIEPT);

			//spInsertExceptionUnitSelection
			DataTable dtIEUS = DAL.ExceptionReport.GetExceptionReportUnitSelection(buGuid).Tables[0].Copy();
			dsSave.Tables.Add(dtIEUS);

			//spInsertExceptionEmergencyIssueUnitTest
			DataTable dtIEEIUT = DAL.ExceptionReport.GetExceptionReportEmergencyIssueUnitTest(UnitTestConfig.TestDivisionCode, patientGuid).Tables[0].Copy();
			dsSave.Tables.Add(dtIEEIUT);

			//spInsertExceptionExpiredTask
			DataTable dtIEET = DAL.ExceptionReport.GetExceptionReportExpiredTask(UnitTestConfig.TestDivisionCode, System.DateTime.Now, System.DateTime.Now).Tables[0].Copy();
			dsSave.Tables.Add(dtIEET);

			//spInsertExceptionInconclusiveCrossmatch
			DataTable dtIEIC = DAL.ExceptionReport.GetExceptionReportInconclusiveCrossmatch(UnitTestConfig.TestDivisionCode, System.DateTime.Now, System.DateTime.Now).Tables[0].Copy();
			dsSave.Tables.Add(dtIEIC);

			//spInsertWorkloadEvent
			System.Guid[] workloadGuids = {patientGuid};
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.Justification);
			DataTable dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.Justification, workloadGuids, Common.WorkloadTransactionType.Patient, false);
			dsSave.Tables.Add(Common.Utility.AppendLastUpdateInformation(dtWorkload, Common.UpdateFunction.UnitTests));

			bool rc = DAL.OrderedTest.SaveTestResults(dsSave);

			Assert.IsTrue(rc, "This method is HUGE!");

		}

		[Test]
		public void SaveTestResults_Fail()
		{
			this.RefreshData = true;

			DataSet dsTest = new DataSet();
			bool retValue = DAL.OrderedTest.SaveTestResults(dsTest);
			
			Assert.AreEqual(false, retValue, "ReturnValue");
		}

		[Test]
		public void UpdateStatus_Pass()
		{
			this.RefreshData = true;

			System.Guid psGuid = System.Guid.NewGuid();
			System.Guid patientGuid = (System.Guid)this._dtPatientSpecimen.Rows[0][TABLES.PatientSpecimen.PatientGuid];
			System.Guid otGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];
			byte[] otRowVersion = (byte[])this._dtOrderedTest.Rows[0][TABLES.OrderedTest.RowVersion];

			DataTable dtInsPS = SprocHelper.GetInsertPatientSpecimenSprocTable(psGuid, patientGuid, DataCreator.GetNewSpecimenUid());
			DataTable dtUpdOT = SprocHelper.GetUpdateOrderedTestStatusSprocTable(otGuid, otRowVersion);

			dtUpdOT.Rows[0][TABLES.OrderedTest.OrderStatusCode] = this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderStatusCode];
			
			System.Guid[] workloadGuids = {otGuid};
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.AcceptOrder);
			DataTable dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.AcceptOrder, workloadGuids, Common.WorkloadTransactionType.Order, false);

			bool retValue = DAL.OrderedTest.UpdateStatus(System.Guid.Empty, 'A', 'B',"C",System.Guid.Empty, "D","E", dtInsPS, dtUpdOT, dtWorkload, Common.UpdateFunction.UnitTests);

			//For 100% compliance we created a second specimen, but now we'll delete it...
			UnitTestUtil.RunSQL("DELETE FROM dbo.PatientSpecimen WHERE PatientSpecimenGuid = '"+ psGuid.ToString() + "'");

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[ExpectedException(typeof(Common.RowVersionException))]
		public void UpdateStatus_Fail()
		{
			this.RefreshData = true;

			System.Guid psGuid = System.Guid.NewGuid();
			System.Guid patientGuid = (System.Guid)this._dtPatientSpecimen.Rows[0][TABLES.PatientSpecimen.PatientGuid];
			System.Guid otGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];
			byte[] otRowVersion = new byte[8];

			DataTable dtInsPS = SprocHelper.GetInsertPatientSpecimenSprocTable(psGuid, patientGuid, DataCreator.GetNewSpecimenUid());
			DataTable dtUpdOT = SprocHelper.GetUpdateOrderedTestStatusSprocTable(otGuid, otRowVersion);

			dtUpdOT.Rows[0][TABLES.OrderedTest.OrderStatusCode] = this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderStatusCode];

			DAL.OrderedTest.UpdateStatus(System.Guid.Empty, 'A', 'B',"C",System.Guid.Empty, "D","E", dtInsPS, dtUpdOT, null, Common.UpdateFunction.UnitTests);
		}

		[Test]
		public void GetOrderData_Pass()
		{
			string cprsOrderNumber = this._dtOrderedTest.Rows[0][TABLES.OrderedTest.CprsOrderNumber].ToString();
			
			DataSet dsTest = DAL.OrderedTest.GetOrderData(cprsOrderNumber);

			Assert.AreEqual(1, dsTest.Tables.Count, "Tables.Count");
			Assert.AreEqual(65, dsTest.Tables[0].Columns.Count, "Columns.Count");

			DataTable dtTest = dsTest.Tables[0];

			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.IsOrderedComponent), ARTIFICIAL.IsOrderedComponent);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.VistaPatientId), TABLES.Patient.VistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDob), TABLES.Patient.PatientDob);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSexCode), TABLES.Patient.PatientSexCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientIcn), TABLES.Patient.PatientIcn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.BloodTypeCode), TABLES.Patient.BloodTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.RhFactorCode), TABLES.Patient.RhFactorCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientNamePrefix), TABLES.Patient.PatientNamePrefix);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientNameSuffix), TABLES.Patient.PatientNameSuffix);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDobCode), TABLES.Patient.PatientDobCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientLocation), TABLES.PatientTreatment.PatientLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientRoomBed), TABLES.PatientTreatment.PatientRoomBed);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientAdmittingDiagnosis), TABLES.PatientTreatment.PatientAdmittingDiagnosis);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientServiceBranch), TABLES.Patient.PatientServiceBranch);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientTreatingFacility), TABLES.PatientTreatment.PatientTreatingFacility);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.InPatientIndicator), TABLES.PatientTreatment.InPatientIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.RecordStatusCode), TABLES.Patient.RecordStatusCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientGuid), TABLES.PatientTreatment.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSsn), TABLES.Patient.PatientSsn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderLastName), TABLES.PatientOrder.OrderingProviderLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderFirstName), TABLES.PatientOrder.OrderingProviderFirstName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderMiddleInitial), TABLES.PatientOrder.OrderingProviderMiddleInitial);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderComments), TABLES.PatientOrder.OrderComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderPlacedDatetime), TABLES.PatientOrder.OrderPlacedDatetime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.PatientOrderGuid), TABLES.PatientOrder.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderGroupNumber), TABLES.PatientOrder.OrderGroupNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderReceivedDatetime), TABLES.PatientOrder.OrderReceivedDatetime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabOrderNumber), TABLES.OrderedTest.LabOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderId), TABLES.PatientOrder.OrderingProviderId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingHospitalLocation), TABLES.PatientOrder.OrderingHospitalLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderEnteredById), TABLES.PatientOrder.OrderEnteredById);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderEnteredByLastName), TABLES.PatientOrder.OrderEnteredByLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderEnteredByFirstName), TABLES.PatientOrder.OrderEnteredByFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderEnteredByMiddleInitial), TABLES.PatientOrder.OrderEnteredByMiddleInitial);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.RequestReason), TABLES.PatientOrder.RequestReason);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.InformedConsentIndicator), TABLES.PatientOrder.InformedConsentIndicator);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.SurgeryName), TABLES.PatientOrder.SurgeryName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.DivisionCode), TABLES.PatientOrder.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestGuid), TABLES.OrderedTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CprsOrderNumber), TABLES.OrderedTest.CprsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.VbecsOrderNumber), TABLES.OrderedTest.VbecsOrderNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.SpecimenRequiredIndicator), TABLES.OrderedTest.SpecimenRequiredIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.PatientSpecimenGuid), TABLES.OrderedTest.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderStatusCode), TABLES.OrderedTest.OrderStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderUrgencyCode), TABLES.OrderedTest.OrderUrgencyCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestComments), TABLES.OrderedTest.OrderedTestComments);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RequiredResultsDate), TABLES.OrderedTest.RequiredResultsDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderableTestId), TABLES.OrderedTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TestInterpretation), TABLES.OrderedTest.TestInterpretation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TestInterpretationComments), TABLES.OrderedTest.TestInterpretationComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.AutoInstrumentName), TABLES.OrderedTest.AutoInstrumentName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TaskStatusCode), TABLES.OrderedTest.TaskStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RelatedTestGuid), TABLES.OrderedTest.RelatedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CancellationCannedCommentGuid), TABLES.OrderedTest.CancellationCannedCommentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.OrderedTestDivisionCode), ARTIFICIAL.OrderedTestDivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestName), TABLES.OrderableTest.OrderableTestName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.CprsOrderableItemId), TABLES.OrderableTest.CprsOrderableItemId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderStatus.OrderStatusText), TABLES.OrderStatus.OrderStatusText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.CannedComment.CannedCommentText), TABLES.CannedComment.CannedCommentText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedComponent.LabOrderNumber), TABLES.OrderedComponent.LabOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedComponent.LabTestId), TABLES.OrderedComponent.LabTestId);
		}

		[Test]
		public void GetOrderData_Fail()
		{
			DataSet dsTest = DAL.OrderedTest.GetOrderData(string.Empty);

			Assert.AreEqual(0, dsTest.Tables[0].Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetQCResultsForOrderedTest_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];

			DataTable dtTest = DAL.OrderedTest.GetQCResultsForOrderedTest(unitGuid);

			Assert.AreEqual(17, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackLot.ReagentGuid), TABLES.RackLot.ReagentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackLot.RackLotGuid), TABLES.RackLot.RackLotGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackLot.ControlCellId), TABLES.RackLot.ControlCellId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackLot.SatisfactoryResultIndicator), TABLES.RackLot.SatisfactoryResultIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackLot.ReactivityPhaseCode), TABLES.RackLot.ReactivityPhaseCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackLot.LotVialNumber), TABLES.RackLot.ReagentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackLot.Comments), TABLES.RackLot.ReagentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Reagent.LotNumber), TABLES.Reagent.LotNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Reagent.LotExpirationDate), TABLES.Reagent.LotExpirationDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Reagent.ReagentTypeId), TABLES.Reagent.ReagentTypeId);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Reagent.ReagentManufacturerName), TABLES.Reagent.ReagentManufacturerName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackTest.RackTestGuid), TABLES.RackTest.RackTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.RackTestRowVersion), ARTIFICIAL.RackTestRowVersion);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackTest.RackTestResult), TABLES.RackTest.RackTestResult);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackTest.RackTestResultTypeCode), TABLES.RackTest.RackTestResultTypeCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackLot.RecordStatusCode), TABLES.RackLot.RecordStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.RackLot.RowVersion), TABLES.RackLot.RowVersion);
		}

		[Test]
		public void GetQCResultsForOrderedTest_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetQCResultsForOrderedTest(System.Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetReagentTypeForOrderableTest_Pass()
		{
			int orderableTestId = (int)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderableTestId];
			DataTable dtTest = DAL.OrderedTest.GetReagentTypeForOrderableTest(orderableTestId);

			Assert.AreEqual(2, dtTest.Columns.Count, "Columns.Count");
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.ReagentType.ReagentTypeId), TABLES.ReagentType.ReagentTypeId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.ReagentType.ReagentTypeName), TABLES.ReagentType.ReagentTypeName);
		}

		[Test]
		public void GetReagentTypeForOrderableTest_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetReagentTypeForOrderableTest(0);
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void InactivateTestResults_Pass()
		{
			this.RefreshData = true;

			System.Guid otGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];
			byte[] stRowVersion = (byte[])this._dtSpecimenTest.Rows[0][TABLES.SpecimenTest.RowVersion];
			byte[] otRowVersion = (byte[])this._dtOrderedTest.Rows[0][TABLES.OrderedTest.RowVersion];

			DataTable dtUpdOT = SprocHelper.GetInactivateOrderedTestResultsSprocTable(otGuid, 1, stRowVersion, new byte[8], new byte[8], new byte[8], otRowVersion);

			dtUpdOT.Rows[0][TABLES.OrderedTest.OrderableTestId] = this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderableTestId];
			dtUpdOT.Rows[0][TABLES.OrderedTest.OrderableTestId] = this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderableTestId];
			dtUpdOT.Rows[0][ARTIFICIAL.RackLotRowVersion] = DBNull.Value;
			dtUpdOT.Rows[0][ARTIFICIAL.RackTestRowVersion] = DBNull.Value;
			dtUpdOT.Rows[0][ARTIFICIAL.AntiseraOrderedTestRowVersion] = DBNull.Value;

			bool retValue = DAL.OrderedTest.InactivateTestResults(dtUpdOT, new DataTable());

			//Reset the data
			if (retValue)
			{
				int uCheck = UnitTestUtil.RunSQL("UPDATE SpecimenTest SET RecordStatusCode = 'A', InactivateDate = NULL, InactivateUser = NULL WHERE SpecimenTestGuid = '" + this._dtSpecimenTest.Rows[0][TABLES.SpecimenTest.SpecimenTestGuid].ToString() + "'");
			}

			Assert.AreEqual(true, retValue, "ReturnValue");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void InactivateTestResults_Fail()
		{
//			Sproc needs RowVersion code for Fail to Pass -- uncomment then

//			DataTable dtOrderedTest = new DataTable(TABLES.OrderedTest.TableName);
//			DataTable dtException = new DataTable(TABLES.ExceptionPatientTestInactivated.TableName);
//
//			dtOrderedTest.Columns.Add(TABLES.OrderedTest.OrderedTestGuid, typeof(System.Guid));
//			dtOrderedTest.Columns.Add(TABLES.SpecimenTest.InactivateDate, typeof(System.DateTime));
//			dtOrderedTest.Columns.Add(TABLES.SpecimenTest.InactivateUser, typeof(string));
//			dtOrderedTest.Columns.Add(TABLES.SpecimenTest.LastUpdateUser, typeof(string));
//			dtOrderedTest.Columns.Add(TABLES.SpecimenTest.DivisionCode, typeof(string));
//			dtOrderedTest.Columns.Add(TABLES.OrderableTest.OrderableTestId, typeof(int));
//			dtOrderedTest.Columns.Add("SpecimenTestRowVersion", typeof(byte[]));
//			dtOrderedTest.Columns.Add("AntiseraOrderedTestRowVersion", typeof(byte[]));
//			dtOrderedTest.Columns.Add("RackLotRowVersion", typeof(byte[]));
//			dtOrderedTest.Columns.Add("RackTestRowVersion", typeof(byte[]));
//			dtOrderedTest.Columns.Add("OrderedTestRowVersion", typeof(byte[]));
//			dtOrderedTest.Columns.Add(TABLES.SpecimenTest.LastUpdateFunctionId, typeof(int));
//
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.ExceptionPatientTestInactivatedGuid, typeof(System.Guid));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.ExceptionTypeId, typeof(int));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.PatientName, typeof(string));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.PatientId, typeof(string));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.SpecimenUid, typeof(string));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.TestName, typeof(string));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.InactivationDate, typeof(System.DateTime));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.CurrentTechName, typeof(string));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.ExceptionComment, typeof(string));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.ExceptionUser, typeof(string));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.ExceptionDate, typeof(System.DateTime));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.TestingTechName, typeof(string));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.DivisionCode, typeof(string));
//			dtException.Columns.Add(TABLES.ExceptionPatientTestInactivated.LastUpdateUser, typeof(string));
//
//			DataRow drTest = dtOrderedTest.NewRow();
//
//			drTest[TABLES.OrderedTest.OrderedTestGuid] = this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];
//			drTest[TABLES.SpecimenTest.InactivateDate] = System.DateTime.Now;
//			drTest[TABLES.SpecimenTest.InactivateUser] = Common.LogonUser.LogonUserName;
//			drTest[TABLES.SpecimenTest.LastUpdateUser] = Common.LogonUser.LogonUserName;
//			drTest[TABLES.SpecimenTest.DivisionCode] = UnitTestConfig.TestDivisionCode;
//			drTest[TABLES.OrderableTest.OrderableTestId] = this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderableTestId];
//			drTest["SpecimenTestRowVersion"] = new byte[8];
//			drTest["AntiseraOrderedTestRowVersion"] = DBNull.Value;
//			drTest["RackLotRowVersion"] = DBNull.Value;
//			drTest["RackTestRowVersion"] = DBNull.Value;
//			drTest["OrderedTestRowVersion"] = this._dtOrderedTest.Rows[0][TABLES.OrderedTest.RowVersion];
//			drTest[TABLES.SpecimenTest.LastUpdateFunctionId] = (int)Common.UpdateFunction.UnitTests;
//
//			dtOrderedTest.Rows.Add(drTest);
//
//			bool retValue = DAL.OrderedTest.InactivateTestResults(dtOrderedTest, dtException);

		}

		[Test]
		public void GetCprsOrderable_Pass()
		{
			DataTable dtTest = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * from OrderableTest", true);

			int orderableTestId = (int)dtTest.Rows[0][TABLES.OrderedTest.OrderableTestId];
			int retValue = DAL.OrderedTest.GetCprsOrderable(orderableTestId);
			
			Assert.AreEqual(true, retValue > 0, "ReturnValue");
		}

		[Test]
		public void GetCprsOrderable_Fail()
		{
			int retValue = DAL.OrderedTest.GetCprsOrderable(int.MaxValue);
			
			Assert.AreEqual(-1, retValue, "Invalid Id");

			DataRow drTemp = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM OrderableTest WHERE CprsOrderableItemID IS NULL");
			int nullCPRSid = System.Convert.ToInt16(drTemp[TABLES.OrderableTest.OrderableTestId]);

			retValue = DAL.OrderedTest.GetCprsOrderable(nullCPRSid);
			Assert.AreEqual(-1, retValue, "Null CPRS Id");
		}

		[Test]
		public void GetOrderedTestForVista_Pass()
		{
			string cprsOrderNumber = this._dtOrderedTest.Rows[0][TABLES.OrderedTest.CprsOrderNumber].ToString();
			DataTable dtTest = DAL.OrderedTest.GetOrderedTestForVista(cprsOrderNumber);

			Assert.AreEqual(65, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.IsOrderedComponent), ARTIFICIAL.IsOrderedComponent);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.VistaPatientId), TABLES.Patient.VistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDob), TABLES.Patient.PatientDob);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSexCode), TABLES.Patient.PatientSexCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientIcn), TABLES.Patient.PatientIcn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.BloodTypeCode), TABLES.Patient.BloodTypeCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.RhFactorCode), TABLES.Patient.RhFactorCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientNamePrefix), TABLES.Patient.PatientNamePrefix);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientNameSuffix), TABLES.Patient.PatientNameSuffix);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientDobCode), TABLES.Patient.PatientDobCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientLocation), TABLES.PatientTreatment.PatientLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientRoomBed), TABLES.PatientTreatment.PatientRoomBed);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientAdmittingDiagnosis), TABLES.PatientTreatment.PatientAdmittingDiagnosis);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientServiceBranch), TABLES.Patient.PatientServiceBranch);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientTreatingFacility), TABLES.PatientTreatment.PatientTreatingFacility);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.InPatientIndicator), TABLES.PatientTreatment.InPatientIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.RecordStatusCode), TABLES.Patient.RecordStatusCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientTreatment.PatientGuid), TABLES.PatientTreatment.PatientGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientSsn), TABLES.Patient.PatientSsn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderLastName), TABLES.PatientOrder.OrderingProviderLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderFirstName), TABLES.PatientOrder.OrderingProviderFirstName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderMiddleInitial), TABLES.PatientOrder.OrderingProviderMiddleInitial);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderComments), TABLES.PatientOrder.OrderComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderPlacedDatetime), TABLES.PatientOrder.OrderPlacedDatetime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.PatientOrderGuid), TABLES.PatientOrder.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderGroupNumber), TABLES.PatientOrder.OrderGroupNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderReceivedDatetime), TABLES.PatientOrder.OrderReceivedDatetime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabOrderNumber), TABLES.OrderedTest.LabOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingProviderId), TABLES.PatientOrder.OrderingProviderId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderingHospitalLocation), TABLES.PatientOrder.OrderingHospitalLocation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderEnteredById), TABLES.PatientOrder.OrderEnteredById);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderEnteredByLastName), TABLES.PatientOrder.OrderEnteredByLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderEnteredByFirstName), TABLES.PatientOrder.OrderEnteredByFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderEnteredByMiddleInitial), TABLES.PatientOrder.OrderEnteredByMiddleInitial);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.RequestReason), TABLES.PatientOrder.RequestReason);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.InformedConsentIndicator), TABLES.PatientOrder.InformedConsentIndicator);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.SurgeryName), TABLES.PatientOrder.SurgeryName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.DivisionCode), TABLES.PatientOrder.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestGuid), TABLES.OrderedTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CprsOrderNumber), TABLES.OrderedTest.CprsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.VbecsOrderNumber), TABLES.OrderedTest.VbecsOrderNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.SpecimenRequiredIndicator), TABLES.OrderedTest.SpecimenRequiredIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.PatientSpecimenGuid), TABLES.OrderedTest.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderStatusCode), TABLES.OrderedTest.OrderStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderUrgencyCode), TABLES.OrderedTest.OrderUrgencyCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestComments), TABLES.OrderedTest.OrderedTestComments);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RequiredResultsDate), TABLES.OrderedTest.RequiredResultsDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderableTestId), TABLES.OrderedTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TestInterpretation), TABLES.OrderedTest.TestInterpretation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TestInterpretationComments), TABLES.OrderedTest.TestInterpretationComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.AutoInstrumentName), TABLES.OrderedTest.AutoInstrumentName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TaskStatusCode), TABLES.OrderedTest.TaskStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RelatedTestGuid), TABLES.OrderedTest.RelatedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CancellationCannedCommentGuid), TABLES.OrderedTest.CancellationCannedCommentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.OrderedTestDivisionCode), ARTIFICIAL.OrderedTestDivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestName), TABLES.OrderableTest.OrderableTestName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.CprsOrderableItemId), TABLES.OrderableTest.CprsOrderableItemId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderStatus.OrderStatusText), TABLES.OrderStatus.OrderStatusText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.CannedComment.CannedCommentText), TABLES.CannedComment.CannedCommentText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedComponent.LabOrderNumber), TABLES.OrderedComponent.LabOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedComponent.LabTestId), TABLES.OrderedComponent.LabTestId);
			
		}

		[Test]
		public void GetOrderedTestForVista_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetOrderedTestForVista(string.Empty);
			
			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}
		
		[Test]
		public void GetHistoricalTestResultsForOrderedTest_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];
			DataTable dtTest = DAL.OrderedTest.GetHistoricalTestResultsForOrderedTest(unitGuid);

			Assert.AreEqual(2, dtTest.Columns.Count, "Columns.Count");
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestResultId), TABLES.SpecimenTest.TestResultId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.BloodTestName), TABLES.BloodTestType.BloodTestName);
		}

		[Test]
		public void GetHistoricalTestResultsForOrderedTest_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetHistoricalTestResultsForOrderedTest(System.Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetOrderedTests_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.PatientSpecimenGuid];
			DataTable dtTest = DAL.OrderedTest.GetOrderedTests(unitGuid);

			Assert.AreEqual(24, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestGuid), TABLES.OrderedTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.PatientOrderGuid), TABLES.OrderedTest.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CprsOrderNumber), TABLES.OrderedTest.CprsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.VbecsOrderNumber), TABLES.OrderedTest.VbecsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RelatedComponentGuid), TABLES.OrderedTest.RelatedComponentGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.SpecimenRequiredIndicator), TABLES.OrderedTest.SpecimenRequiredIndicator);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.PatientSpecimenGuid), TABLES.OrderedTest.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderStatusCode), TABLES.OrderedTest.OrderStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderUrgencyCode), TABLES.OrderedTest.OrderUrgencyCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestComments), TABLES.OrderedTest.OrderedTestComments);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RequiredResultsDate), TABLES.OrderedTest.RequiredResultsDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderableTestId), TABLES.OrderedTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TestInterpretation), TABLES.OrderedTest.TestInterpretation);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TestInterpretationComments), TABLES.OrderedTest.TestInterpretationComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.AutoInstrumentName), TABLES.OrderedTest.AutoInstrumentName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TaskStatusCode), TABLES.OrderedTest.TaskStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RelatedTestGuid), TABLES.OrderedTest.RelatedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CancellationCannedCommentGuid), TABLES.OrderedTest.CancellationCannedCommentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CancellationText), TABLES.OrderedTest.CancellationText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabOrderNumber), TABLES.OrderedTest.LabOrderNumber);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabTestId), TABLES.OrderedTest.LabTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.DivisionCode), TABLES.OrderedTest.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LastUpdateDate), TABLES.OrderedTest.LastUpdateDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LastUpdateUser), TABLES.OrderedTest.LastUpdateUser);
		}

		[Test]
		public void GetOrderedTests_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetOrderedTests(System.Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetRecentTestOrders_Pass()
		{
			DataSet dsTest = DAL.OrderedTest.GetRecentTestOrders(System.Guid.NewGuid());

			Assert.AreEqual(2, dsTest.Tables.Count, "Tables.Count");
			Assert.AreEqual(16, dsTest.Tables[0].Columns.Count, "Tables[0].Columns.Count");
			Assert.AreEqual(8, dsTest.Tables[1].Columns.Count, "Tables[1].Columns.Count");

			DataTable dtTest = dsTest.Tables[0];
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestGuid), "[0]-" + TABLES.OrderedTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.PatientOrderGuid), "[0]-" + TABLES.OrderedTest.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.OrderReceivedDatetime), "[0]-" + TABLES.PatientOrder.OrderReceivedDatetime);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderableTestId), "[0]-" + TABLES.OrderedTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestText), "[0]-" + TABLES.OrderableTest.OrderableTestText);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.RequiredResultsDate), "[0]-" + TABLES.OrderedTest.RequiredResultsDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CprsOrderNumber), "[0]-" + TABLES.OrderedTest.CprsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderStatusCode), "[0]-" + TABLES.OrderedTest.OrderStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderStatus.OrderStatusText), "[0]-" + TABLES.OrderStatus.OrderStatusText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.TaskStatusCode), "[0]-" + TABLES.OrderedTest.TaskStatusCode);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TaskStatus.TaskStatusText), "[0]-" + TABLES.TaskStatus.TaskStatusText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestComments), "[0]-" + TABLES.OrderedTest.OrderedTestComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.DivisionCode), "[0]-" + TABLES.OrderedTest.DivisionCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.VamcDivision.DivisionName), "[0]-" + TABLES.VamcDivision.DivisionName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.PatientSpecimenGuid), "[0]-" + TABLES.PatientSpecimen.PatientSpecimenGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.SpecimenUid), "[0]-" + TABLES.PatientSpecimen.SpecimenUid);

			dtTest = dsTest.Tables[1];
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestGuid), "[1]-" + TABLES.OrderedTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.PatientOrderGuid), "[1]-" + TABLES.OrderedTest.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderableTestId), "[1]-" + TABLES.OrderedTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.ReportableTestId), "[1]-" + TABLES.BloodTestType.ReportableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestText), "[1]-" + TABLES.OrderableTest.OrderableTestText);

			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.Interp), "[1]-" + ARTIFICIAL.Interp);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestDate), "[1]-" + TABLES.SpecimenTest.TestDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestComments), "[1]-" + TABLES.SpecimenTest.TestComments);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void GetRecentTestOrders_Fail()
		{
			DAL.OrderedTest.GetRecentTestOrders(System.Guid.Empty);
		}

		[Test]
		public void GetRecentTestOrdersCount_Pass()
		{
			System.Guid patientGuid = (System.Guid)this._dtPatientSpecimen.Rows[0][TABLES.PatientSpecimen.PatientSpecimenGuid];

			Assert.IsTrue(DAL.OrderedTest.GetRecentTestOrdersCount(patientGuid)>=0);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetRecentTestOrdersCount_Fail()
		{
			Assert.IsTrue(DAL.OrderedTest.GetRecentTestOrdersCount(Guid.Empty)==0);
		}

		[Test]
		public void GetTestOrders_Guid_Pass()
		{
			System.Guid unitGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.PatientSpecimenGuid];

			DataTable dtTest = DAL.OrderedTest.GetTestOrders(unitGuid);

			Assert.AreEqual(16, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderedTestGuid), TABLES.OrderedTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedComponent.OrderedComponentGuid), TABLES.OrderedComponent.OrderedComponentGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.BloodUnitGuid), TABLES.BloodUnit.BloodUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.SpecimenTestGuid), TABLES.SpecimenTest.SpecimenTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.BloodUnitTestGuid), TABLES.BloodUnitTest.BloodUnitTestGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.PatientOrderGuid), TABLES.PatientOrder.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.OrderableTestId), TABLES.OrderedTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.ReportableTestId), TABLES.BloodTestType.ReportableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestText), TABLES.OrderableTest.OrderableTestText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.EyeReadableUnitId), TABLES.BloodUnit.EyeReadableUnitId);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.PatientOrderGuid), TABLES.PatientOrder.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestId), TABLES.OrderableTest.OrderableTestId);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.Interp), ARTIFICIAL.Interp);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestTechId), TABLES.SpecimenTest.TestTechId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestDate), TABLES.SpecimenTest.TestDate);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.AboRhChangeIndicator), TABLES.Patient.AboRhChangeIndicator);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetTestOrders_Guid_Fail()
		{
			DAL.OrderedTest.GetTestOrders(System.Guid.Empty);
		}

		[Test]
		public void GetSpecimenTestResultsForOrderedTest_Guid_Pass()
		{
			DataTable dtTest = DAL.OrderedTest.GetSpecimenTestResultsForOrderedTest(System.Guid.NewGuid());

			Assert.AreEqual(27, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.PatientSpecimenGuid), TABLES.PatientSpecimen.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.OrderedTestGuid), TABLES.SpecimenTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.SpecimenTestGuid), TABLES.SpecimenTest.SpecimenTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.PatientOrderGuid), TABLES.PatientOrder.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.ReportableTestId), TABLES.BloodTestType.ReportableTestId);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestText), TABLES.OrderableTest.OrderableTestText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestingMethodCode), TABLES.SpecimenTest.TestingMethodCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TestingMethod.TestingMethodText), TABLES.TestingMethod.TestingMethodText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.RackGuid), TABLES.SpecimenTest.RackGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Rack.RackName), TABLES.Rack.RackName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.TestedWith), TABLES.BloodTestType.TestedWith);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.BloodTestTypeId), TABLES.SpecimenTest.BloodTestTypeId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.BloodTestName), TABLES.BloodTestType.BloodTestName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.Phase), TABLES.BloodTestType.Phase);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.SortColumn), TABLES.BloodTestType.SortColumn);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestResultId), TABLES.SpecimenTest.TestResultId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TestResult.TestResultText), TABLES.TestResult.TestResultText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.AntibodyTypeId), TABLES.SpecimenTest.AntibodyTypeId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.AntibodyType.AntibodyTypeName), TABLES.AntibodyType.AntibodyTypeName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestTechId), TABLES.SpecimenTest.TestTechId);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestDate), TABLES.SpecimenTest.TestDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestComments), TABLES.SpecimenTest.TestComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.AutoInstrumentName), TABLES.SpecimenTest.AutoInstrumentName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.RecordStatusCode), TABLES.SpecimenTest.RecordStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.RowVersion), TABLES.SpecimenTest.RowVersion);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.AboRhChangeIndicator), TABLES.SpecimenTest.AboRhChangeIndicator);
            Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PendingSpecimenTest.SpecimenResultSetGuid), TABLES.PendingSpecimenTest.SpecimenResultSetGuid);
			
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetSpecimenTestResultsForOrderedTest_Guid_Fail()
		{
			DAL.OrderedTest.GetSpecimenTestResultsForOrderedTest(System.Guid.Empty);
		}

		[Test]
		public void GetSpecimenTestResultsForOrderedTest_Guid_Int_Pass()
		{
			DataTable dtTest = DAL.OrderedTest.GetSpecimenTestResultsForOrderedTest(System.Guid.NewGuid(), 1);
			
			Assert.AreEqual(27, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.PatientSpecimenGuid), TABLES.PatientSpecimen.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.OrderedTestGuid), TABLES.SpecimenTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.SpecimenTestGuid), TABLES.SpecimenTest.SpecimenTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientOrder.PatientOrderGuid), TABLES.PatientOrder.PatientOrderGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.ReportableTestId), TABLES.BloodTestType.ReportableTestId);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderableTest.OrderableTestText), TABLES.OrderableTest.OrderableTestText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestingMethodCode), TABLES.SpecimenTest.TestingMethodCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TestingMethod.TestingMethodText), TABLES.TestingMethod.TestingMethodText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.RackGuid), TABLES.SpecimenTest.RackGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Rack.RackName), TABLES.Rack.RackName);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.TestedWith), TABLES.BloodTestType.TestedWith);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.BloodTestTypeId), TABLES.SpecimenTest.BloodTestTypeId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.BloodTestName), TABLES.BloodTestType.BloodTestName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.Phase), TABLES.BloodTestType.Phase);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.SortColumn), TABLES.BloodTestType.SortColumn);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestResultId), TABLES.SpecimenTest.TestResultId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TestResult.TestResultText), TABLES.TestResult.TestResultText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.AntibodyTypeId), TABLES.SpecimenTest.AntibodyTypeId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.AntibodyType.AntibodyTypeName), TABLES.AntibodyType.AntibodyTypeName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestTechId), TABLES.SpecimenTest.TestTechId);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestDate), TABLES.SpecimenTest.TestDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.TestComments), TABLES.SpecimenTest.TestComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.AutoInstrumentName), TABLES.SpecimenTest.AutoInstrumentName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.RecordStatusCode), TABLES.SpecimenTest.RecordStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.RowVersion), TABLES.SpecimenTest.RowVersion);
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.SpecimenTest.AboRhChangeIndicator), TABLES.SpecimenTest.AboRhChangeIndicator);
            Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PendingSpecimenTest.SpecimenResultSetGuid), TABLES.PendingSpecimenTest.SpecimenResultSetGuid);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetSpecimenTestResultsForOrderedTest_Guid_Int_Fail()
		{
			DAL.OrderedTest.GetSpecimenTestResultsForOrderedTest(System.Guid.Empty, 1);
		}

		[Test]
		public void GetBloodUnitTestResultsForOrderedTest_Guid_Pass()
		{
			DataTable dtTest = DAL.OrderedTest.GetBloodUnitTestResultsForOrderedTest(System.Guid.NewGuid());
			
			Assert.AreEqual(27, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.PatientSpecimenGuid), TABLES.PatientSpecimen.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.OrderedTestGuid), TABLES.BloodUnitTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.BloodUnitTestGuid), TABLES.BloodUnitTest.BloodUnitTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.OrderedUnitGuid), TABLES.BloodUnitTest.OrderedUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.BloodUnitGuid), TABLES.BloodUnitTest.BloodUnitGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.EyeReadableUnitId), TABLES.BloodUnit.EyeReadableUnitId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductShortName), TABLES.BloodProduct.ProductShortName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.TestingMethodCode), TABLES.BloodUnitTest.TestingMethodCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TestingMethod.TestingMethodText), TABLES.TestingMethod.TestingMethodText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.RackGuid), TABLES.BloodUnitTest.RackGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Rack.RackName), TABLES.Rack.RackName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.TestedWith), TABLES.BloodTestType.TestedWith);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.BloodTestTypeId), TABLES.BloodTestType.BloodTestTypeId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.BloodTestName), TABLES.BloodTestType.BloodTestName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.Phase), TABLES.BloodTestType.Phase);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.SortColumn), TABLES.BloodTestType.SortColumn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.TestResultId), TABLES.BloodUnitTest.TestResultId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TestResult.TestResultText), TABLES.TestResult.TestResultText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.TestTechId), TABLES.BloodUnitTest.TestTechId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.TestDate), TABLES.BloodUnitTest.TestDate);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.TestComments), TABLES.BloodUnitTest.TestComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.AutoInstrumentName), TABLES.BloodUnitTest.AutoInstrumentName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.RecordStatusCode), TABLES.BloodUnitTest.RecordStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.InactivateUser), TABLES.BloodUnitTest.InactivateUser);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.InactivateDate), TABLES.BloodUnitTest.InactivateDate);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.RowVersion), TABLES.BloodUnitTest.RowVersion);
            Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PendingBloodUnitTest.UnitResultSetGuid), TABLES.PendingBloodUnitTest.UnitResultSetGuid);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetBloodUnitTestResultsForOrderedTest_Guid_Fail()
		{
			DAL.OrderedTest.GetBloodUnitTestResultsForOrderedTest(System.Guid.Empty);
		}

		[Test]
		public void GetBloodUnitTestResultsForOrderedTest_Guid_Guid_Pass()
		{
			DataTable dtTest = DAL.OrderedTest.GetBloodUnitTestResultsForOrderedTest(System.Guid.NewGuid(), System.Guid.NewGuid());
			
			Assert.AreEqual(27, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PatientSpecimen.PatientSpecimenGuid), TABLES.PatientSpecimen.PatientSpecimenGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.OrderedTestGuid), TABLES.BloodUnitTest.OrderedTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.BloodUnitTestGuid), TABLES.BloodUnitTest.BloodUnitTestGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.OrderedUnitGuid), TABLES.BloodUnitTest.OrderedUnitGuid);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.BloodUnitGuid), TABLES.BloodUnitTest.BloodUnitGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnit.EyeReadableUnitId), TABLES.BloodUnit.EyeReadableUnitId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodProduct.ProductShortName), TABLES.BloodProduct.ProductShortName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.TestingMethodCode), TABLES.BloodUnitTest.TestingMethodCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TestingMethod.TestingMethodText), TABLES.TestingMethod.TestingMethodText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.RackGuid), TABLES.BloodUnitTest.RackGuid);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Rack.RackName), TABLES.Rack.RackName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.TestedWith), TABLES.BloodTestType.TestedWith);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.BloodTestTypeId), TABLES.BloodTestType.BloodTestTypeId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.BloodTestName), TABLES.BloodTestType.BloodTestName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.Phase), TABLES.BloodTestType.Phase);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodTestType.SortColumn), TABLES.BloodTestType.SortColumn);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.TestResultId), TABLES.BloodUnitTest.TestResultId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.TestResult.TestResultText), TABLES.TestResult.TestResultText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.TestTechId), TABLES.BloodUnitTest.TestTechId);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.TestDate), TABLES.BloodUnitTest.TestDate);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.TestComments), TABLES.BloodUnitTest.TestComments);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.AutoInstrumentName), TABLES.BloodUnitTest.AutoInstrumentName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.RecordStatusCode), TABLES.BloodUnitTest.RecordStatusCode);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.InactivateUser), TABLES.BloodUnitTest.InactivateUser);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.InactivateDate), TABLES.BloodUnitTest.InactivateDate);

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.BloodUnitTest.RowVersion), TABLES.BloodUnitTest.RowVersion);
            Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.PendingBloodUnitTest.UnitResultSetGuid), TABLES.PendingBloodUnitTest.UnitResultSetGuid);
		}

		[Test]
		public void GetBloodUnitTestResultsForOrderedTest_Guid_Guid_Fail()
		{
			try
			{
				DAL.OrderedTest.GetBloodUnitTestResultsForOrderedTest(System.Guid.Empty, System.Guid.NewGuid());
				Assert.AreEqual(true, false, "1-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.AreEqual(true, true, "Empty OrderedTestGuid");
			}
			try
			{
				DAL.OrderedTest.GetBloodUnitTestResultsForOrderedTest(System.Guid.NewGuid(), System.Guid.Empty);
				Assert.AreEqual(true, false, "2-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.AreEqual(true, true, "Empty BloodUnitGuid");
			}
		}

		[Test]
		public void GetRequiredOrders_Pass()
		{
			DataTable dtTest = DAL.OrderedTest.GetRequiredOrders(UnitTestConfig.TestDivisionCode, 24, System.DateTime.Now.AddYears(-1));

			Assert.AreEqual(11, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderUrgency.OrderUrgencyText), TABLES.OrderUrgency.OrderUrgencyText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);

			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.OrderedItem), ARTIFICIAL.OrderedItem);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.WantedDate), ARTIFICIAL.WantedDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CprsOrderNumber), TABLES.OrderedTest.CprsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabOrderNumber), TABLES.OrderedTest.LabOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.UrgencySortColumn), Common.DatabaseConstants.ArtificialColumnNames.UrgencySortColumn);
			Assert.AreEqual(true, dtTest.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.RequestSortColumn), Common.DatabaseConstants.ArtificialColumnNames.RequestSortColumn);

		}

		[Test]
		public void GetRequiredOrders_Fail()
		{
			try
			{
				DAL.OrderedTest.GetRequiredOrders(null, 1, System.DateTime.Now);
				Assert.AreEqual(true, false, "1-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch (ArgumentException)
			{
				Assert.AreEqual(true, true, "Null Division");
			}
			try
			{
				DAL.OrderedTest.GetRequiredOrders(string.Empty, 1, System.DateTime.Now);
				Assert.AreEqual(true, false, "2-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.AreEqual(true, true, "Empty Division");
			}
			try
			{
				DAL.OrderedTest.GetRequiredOrders(UnitTestConfig.TestDivisionCode, -1, System.DateTime.Now);
				Assert.AreEqual(true, false, "3-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.AreEqual(true, true, "Invalid Hours");
			}
		}

		[Test]
		public void GetOrdersWithSpecimenInBloodBank_Pass()
		{
			this.RefreshData = true;

			Common.LogonUser.LogonUserDivisionCode = UnitTestConfig.TestDivisionCode;
			DateTime lastChecked = VbecsDateTime.GetDivisionCurrentDateTime().Subtract(new TimeSpan(777, 0, 0, 0, 0));
			DataTable dtOrdersWithSpecimens = DAL.OrderedTest.GetOrdersWithSpecimenInBloodBank(UnitTestConfig.TestDivisionCode, lastChecked);

			if (dtOrdersWithSpecimens.Rows.Count == 0)
			{
				StringBuilder query = new StringBuilder("SELECT TOP 1 PS.PatientSpecimenGuid, OT.OrderedTestGuid ");
				query.Append("FROM OrderedTest AS OT ");
				query.Append("JOIN OrderUrgency AS OU ON OT.OrderUrgencyCode = OU.OrderUrgencyCode ");
				query.Append("JOIN OrderableTest OTE ON OT.OrderableTestId = OTE.OrderableTestId ");
				query.Append("JOIN PatientOrder AS PO ON OT.PatientOrderGuid = PO.PatientOrderGuid ");
				query.Append("JOIN PatientTreatment AS PT ON PO.PatientTreatmentGuid = PT.PatientTreatmentGuid ");
				query.Append("JOIN Patient AS P ON PT.PatientGuid = P.PatientGuid ");
				query.Append("JOIN PatientSpecimen AS PS ON P.PatientGuid = PS.PatientGuid ");
				query.Append("WHERE OT.OrderStatusCode = 'P' ");
				query.Append("AND OT.DivisionCode = '").Append(UnitTestConfig.TestDivisionCode).Append("' ");
				query.Append("AND OT.LastUpdateDate > '").Append(lastChecked).Append("' ");

				DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable(query.ToString(), true);

				System.Guid patientSpecimenGuid = (System.Guid) dtTemp.Rows[0][TABLES.OrderedTest.PatientSpecimenGuid];
				System.Guid orderedTestGuid = (System.Guid) dtTemp.Rows[0][TABLES.OrderedTest.OrderedTestGuid];
				

				query.Remove(0, query.Length);
				query.Append("UPDATE OrderedTest SET PatientSpecimenGuid = '");
				query.Append(patientSpecimenGuid);
				query.Append("' WHERE OrderedTestGuid = '");
				query.Append(orderedTestGuid);
				query.Append("'");

				int uCheck = UnitTestUtil.RunSQL(query.ToString());

				// Try again
				dtOrdersWithSpecimens = DAL.OrderedTest.GetOrdersWithSpecimenInBloodBank(UnitTestConfig.TestDivisionCode, lastChecked);
			}
			
			Assert.AreEqual(true, dtOrdersWithSpecimens.Rows.Count > 0, "NO new orders having no required specimen or specimen on file date since " + lastChecked.ToLongDateString());
		}

		[Test]
		public void GetOrdersWithSpecimenInBloodBank_Fail()
		{
			try
			{
				DAL.OrderedTest.GetOrdersWithSpecimenInBloodBank(null, System.DateTime.Now);
				Assert.AreEqual(true, false, "1" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch (ArgumentException)
			{
				Assert.AreEqual(true, true, "Null Division");
			}
			try
			{
				DAL.OrderedTest.GetOrdersWithSpecimenInBloodBank(string.Empty, System.DateTime.Now);
				Assert.AreEqual(true, false, "2-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.AreEqual(true, true, "Empty Division");
			}
		}

		[Test]
		public void GetNewOrders_Pass()
		{
			DataTable dtTest = DAL.OrderedTest.GetNewOrders(UnitTestConfig.TestDivisionCode, System.DateTime.Now.AddYears(-1));
			
			Assert.AreEqual(11, dtTest.Columns.Count, "Columns.Count");

			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderUrgency.OrderUrgencyText), TABLES.OrderUrgency.OrderUrgencyText);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientLastName), TABLES.Patient.PatientLastName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientFirstName), TABLES.Patient.PatientFirstName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.PatientMiddleName), TABLES.Patient.PatientMiddleName);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.Patient.DisplayVistaPatientId), TABLES.Patient.DisplayVistaPatientId);

			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.OrderedItem), ARTIFICIAL.OrderedItem);
			Assert.AreEqual(true, dtTest.Columns.Contains(ARTIFICIAL.WantedDate), ARTIFICIAL.WantedDate);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.CprsOrderNumber), TABLES.OrderedTest.CprsOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedTest.LabOrderNumber), TABLES.OrderedTest.LabOrderNumber);
			Assert.AreEqual(true, dtTest.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.UrgencySortColumn), Common.DatabaseConstants.ArtificialColumnNames.UrgencySortColumn);
			Assert.AreEqual(true, dtTest.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.RequestSortColumn), Common.DatabaseConstants.ArtificialColumnNames.RequestSortColumn);

		}

		[Test]
		public void GetNewOrders_Fail()
		{
			try
			{
				DAL.OrderedTest.GetNewOrders(null, System.DateTime.Now);
				Assert.AreEqual(true, false, "1-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch (ArgumentException)
			{
				Assert.AreEqual(true, true, "Null Division");
			}
			try
			{
				DAL.OrderedTest.GetNewOrders(string.Empty, System.DateTime.Now);
				Assert.AreEqual(true, false, "2-" + AssertionFailReasons.ArgumentExceptionExpected);
			}
			catch(ArgumentException)
			{
				Assert.AreEqual(true, true, "Empty Division");
			}
		}

		[Test]
		public void GetEmergencyIssuedUnitsForOrderedTest_Pass()
		{
			System.Guid otGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];

			DataTable dtTest = DAL.OrderedTest.GetEmergencyIssuedUnitsForOrderedTest(otGuid);

			Assert.AreEqual(1, dtTest.Columns.Count, "Columns.Count");
			
			Assert.AreEqual(true, dtTest.Columns.Contains(TABLES.OrderedUnit.BloodUnitGuid), TABLES.OrderedUnit.BloodUnitGuid);

		}

		[Test]
		public void GetEmergencyIssuedUnitsForOrderedTest_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetEmergencyIssuedUnitsForOrderedTest(System.Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Columns.Count");
		}

		[Test]
		public void GetAntiseraOrderedTests_Pass()
		{
			System.Guid otGuid = (System.Guid)this._dtOrderedTest.Rows[0][TABLES.OrderedTest.OrderedTestGuid];

			DataTable dtTest = DAL.OrderedTest.GetAntiseraOrderedTests(otGuid);

			Assert.AreEqual(7, dtTest.Columns.Count, "Columns.Count");
		}

		[Test]
		public void GetAntiseraOrderedTests_Fail()
		{
			DataTable dtTest = DAL.OrderedTest.GetAntiseraOrderedTests(System.Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Columns.Count");
		}
	}
}
#endif
